@extends('layouts.app')

@section('title', 'Manage Doctors')

@section('content')
<div class="max-w-7xl mx-auto px-2 sm:px-4 lg:px-8">
    <div class="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-4 sm:mb-6 gap-3">
        <h1 class="text-2xl sm:text-3xl font-bold text-gray-800">Manage Doctors</h1>
        <div class="flex gap-2">
            <a href="{{ route('admin.doctors.create') }}" class="bg-blue-600 hover:bg-blue-700 text-white px-3 sm:px-4 py-2 rounded text-sm sm:text-base">
                <i class="fas fa-plus"></i> <span class="hidden sm:inline">Register Doctor</span><span class="sm:hidden">Add</span>
            </a>
            <a href="{{ route('admin.dashboard') }}" class="bg-gray-600 hover:bg-gray-700 text-white px-3 sm:px-4 py-2 rounded text-sm sm:text-base">
                <i class="fas fa-arrow-left"></i> <span class="hidden sm:inline">Back to Dashboard</span><span class="sm:hidden">Back</span>
            </a>
        </div>
    </div>

    @if(session('success'))
        <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
            {{ session('success') }}
        </div>
    @endif

    <!-- Quick Filter Buttons -->
    <div class="mb-4 flex flex-wrap gap-2">
        @php
            $preserveParams = request()->only(['search', 'sort_by', 'sort_order']);
        @endphp
        <a href="{{ route('admin.doctors', $preserveParams) }}" 
           class="px-3 py-1 rounded-full text-sm {{ !request()->has('status') ? 'bg-blue-600 text-white' : 'bg-gray-200 text-gray-700 hover:bg-gray-300' }}">
            All ({{ $totalCount }})
        </a>
        <a href="{{ route('admin.doctors', array_merge($preserveParams, ['status' => 'pending'])) }}" 
           class="px-3 py-1 rounded-full text-sm {{ request('status') === 'pending' ? 'bg-red-600 text-white' : 'bg-gray-200 text-gray-700 hover:bg-gray-300' }}">
            Pending ({{ $pendingCount }})
        </a>
        <a href="{{ route('admin.doctors', array_merge($preserveParams, ['status' => 'active'])) }}" 
           class="px-3 py-1 rounded-full text-sm {{ request('status') === 'active' ? 'bg-green-600 text-white' : 'bg-gray-200 text-gray-700 hover:bg-gray-300' }}">
            Active ({{ $activeCount }})
        </a>
        <a href="{{ route('admin.doctors', array_merge($preserveParams, ['status' => 'inactive'])) }}" 
           class="px-3 py-1 rounded-full text-sm {{ request('status') === 'inactive' ? 'bg-yellow-600 text-white' : 'bg-gray-200 text-gray-700 hover:bg-gray-300' }}">
            Inactive ({{ $inactiveCount }})
        </a>
    </div>

    <!-- Search, Sort, and Filter Section -->
    <div class="bg-white rounded-lg shadow p-4 sm:p-6 mb-4 sm:mb-6">
        <form method="GET" action="{{ route('admin.doctors') }}" class="space-y-4">
            <!-- Search Bar -->
            <div class="flex flex-col sm:flex-row gap-3">
                <div class="flex-1">
                    <label class="block text-sm font-medium text-gray-700 mb-1">Search Doctors</label>
                    <div class="relative">
                        <input type="text" 
                               name="search" 
                               value="{{ request('search') }}" 
                               placeholder="Search by name, email, practice number, specialization..." 
                               class="w-full pl-10 pr-4 py-2 border rounded-lg text-sm sm:text-base">
                        <i class="fas fa-search absolute left-3 top-3 text-gray-400"></i>
                        @if(request('search'))
                            <button type="button" 
                                    onclick="document.querySelector('input[name=search]').value=''; this.form.submit();" 
                                    class="absolute right-3 top-3 text-gray-400 hover:text-gray-600">
                                <i class="fas fa-times"></i>
                            </button>
                        @endif
                    </div>
                </div>
            </div>

            <!-- Filters and Sort -->
            <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-3">
                <!-- Status Filter -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Filter by Status</label>
                    <select name="status" class="w-full px-3 py-2 border rounded-lg text-sm sm:text-base">
                        <option value="">All Status</option>
                        <option value="pending" {{ request('status') === 'pending' ? 'selected' : '' }}>
                            Pending
                        </option>
                        <option value="verified" {{ request('status') === 'verified' ? 'selected' : '' }}>
                            Verified
                        </option>
                        <option value="active" {{ request('status') === 'active' ? 'selected' : '' }}>
                            Active
                        </option>
                        <option value="inactive" {{ request('status') === 'inactive' ? 'selected' : '' }}>
                            Inactive
                        </option>
                    </select>
                </div>

                <!-- Sort By -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Sort By</label>
                    <select name="sort_by" class="w-full px-3 py-2 border rounded-lg text-sm sm:text-base">
                        <option value="created_at" {{ request('sort_by', 'created_at') === 'created_at' ? 'selected' : '' }}>Date Registered</option>
                        <option value="name" {{ request('sort_by') === 'name' ? 'selected' : '' }}>Name (A-Z)</option>
                        <option value="email" {{ request('sort_by') === 'email' ? 'selected' : '' }}>Email</option>
                        <option value="practice_number" {{ request('sort_by') === 'practice_number' ? 'selected' : '' }}>Practice Number</option>
                        <option value="status" {{ request('sort_by') === 'status' ? 'selected' : '' }}>Status</option>
                    </select>
                </div>

                <!-- Sort Order -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Order</label>
                    <select name="sort_order" class="w-full px-3 py-2 border rounded-lg text-sm sm:text-base">
                        <option value="desc" {{ request('sort_order', 'desc') === 'desc' ? 'selected' : '' }}>Descending</option>
                        <option value="asc" {{ request('sort_order') === 'asc' ? 'selected' : '' }}>Ascending</option>
                    </select>
                </div>

                <!-- Action Buttons -->
                <div class="flex items-end gap-2">
                    <button type="submit" class="flex-1 bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg text-sm sm:text-base">
                        <i class="fas fa-filter"></i> <span class="hidden sm:inline">Apply</span><span class="sm:hidden">Filter</span>
                    </button>
                    @if(request()->hasAny(['search', 'status', 'sort_by', 'sort_order']))
                        <a href="{{ route('admin.doctors') }}" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 rounded-lg text-sm sm:text-base">
                            <i class="fas fa-times"></i> <span class="hidden sm:inline">Clear</span>
                        </a>
                    @endif
                </div>
            </div>
        </form>
    </div>

    <!-- Results Count -->
    @if(request()->hasAny(['search', 'status', 'sort_by', 'sort_order']))
        <div class="mb-4 text-sm text-gray-600">
            Showing {{ $doctors->count() }} doctor(s)
            @if(request('search'))
                matching "{{ request('search') }}"
            @endif
        </div>
    @endif

    <div class="bg-white rounded-lg shadow overflow-hidden">
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        @php
                            $currentSort = request('sort_by', 'created_at');
                            $currentOrder = request('sort_order', 'desc');
                            
                            function buildSortUrl($sortBy, $currentSort, $currentOrder) {
                                $params = request()->except(['sort_by', 'sort_order']);
                                $params['sort_by'] = $sortBy;
                                $params['sort_order'] = ($currentSort === $sortBy && $currentOrder === 'asc') ? 'desc' : 'asc';
                                return route('admin.doctors', $params);
                            }
                        @endphp
                        <th class="px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                            <a href="{{ buildSortUrl('name', $currentSort, $currentOrder) }}" 
                               class="flex items-center hover:text-gray-700">
                                Name
                                @if($currentSort === 'name')
                                    <i class="fas fa-sort-{{ $currentOrder === 'asc' ? 'up' : 'down' }} ml-1 text-blue-600"></i>
                                @else
                                    <i class="fas fa-sort ml-1 text-gray-400"></i>
                                @endif
                            </a>
                        </th>
                        <th class="px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden md:table-cell">
                            <a href="{{ buildSortUrl('email', $currentSort, $currentOrder) }}" 
                               class="flex items-center hover:text-gray-700">
                                Email
                                @if($currentSort === 'email')
                                    <i class="fas fa-sort-{{ $currentOrder === 'asc' ? 'up' : 'down' }} ml-1 text-blue-600"></i>
                                @else
                                    <i class="fas fa-sort ml-1 text-gray-400"></i>
                                @endif
                            </a>
                        </th>
                        <th class="px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                            <a href="{{ buildSortUrl('practice_number', $currentSort, $currentOrder) }}" 
                               class="flex items-center hover:text-gray-700">
                                Practice Number
                                @if($currentSort === 'practice_number')
                                    <i class="fas fa-sort-{{ $currentOrder === 'asc' ? 'up' : 'down' }} ml-1 text-blue-600"></i>
                                @else
                                    <i class="fas fa-sort ml-1 text-gray-400"></i>
                                @endif
                            </a>
                        </th>
                        <th class="px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                            <a href="{{ buildSortUrl('status', $currentSort, $currentOrder) }}" 
                               class="flex items-center hover:text-gray-700">
                                Status
                                @if($currentSort === 'status')
                                    <i class="fas fa-sort-{{ $currentOrder === 'asc' ? 'up' : 'down' }} ml-1 text-blue-600"></i>
                                @else
                                    <i class="fas fa-sort ml-1 text-gray-400"></i>
                                @endif
                            </a>
                        </th>
                        <th class="px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @forelse($doctors as $doctor)
                    <tr class="hover:bg-gray-50 cursor-pointer" onclick="window.location='{{ route('admin.doctors.show', $doctor->id) }}'">
                        <td class="px-3 sm:px-6 py-4 whitespace-nowrap">
                            <div class="flex items-center">
                                @if($doctor->profile_picture_path)
                                    <img src="{{ asset('storage/' . $doctor->profile_picture_path) }}" 
                                         alt="{{ $doctor->full_name }}" 
                                         class="h-10 w-10 rounded-full object-cover mr-3 hidden sm:block">
                                @else
                                    <div class="h-10 w-10 rounded-full bg-gray-300 flex items-center justify-center mr-3 hidden sm:block">
                                        <i class="fas fa-user text-gray-600"></i>
                                    </div>
                                @endif
                                <span class="text-sm sm:text-base font-medium">{{ $doctor->full_name }}</span>
                            </div>
                        </td>
                        <td class="px-3 sm:px-6 py-4 whitespace-nowrap hidden md:table-cell">
                            <span class="text-sm">{{ $doctor->email }}</span>
                        </td>
                        <td class="px-3 sm:px-6 py-4 whitespace-nowrap">
                            <span class="text-sm">{{ $doctor->practice_number }}</span>
                        </td>
                        <td class="px-3 sm:px-6 py-4 whitespace-nowrap">
                            @if($doctor->is_verified && $doctor->is_active)
                                <span class="px-2 py-1 text-xs rounded-full bg-green-100 text-green-800">Active</span>
                            @elseif($doctor->is_verified && !$doctor->is_active)
                                <span class="px-2 py-1 text-xs rounded-full bg-yellow-100 text-yellow-800">Inactive</span>
                            @else
                                <span class="px-2 py-1 text-xs rounded-full bg-red-100 text-red-800">Pending</span>
                            @endif
                        </td>
                        <td class="px-3 sm:px-6 py-4 whitespace-nowrap text-sm font-medium" onclick="event.stopPropagation()">
                            <div class="flex flex-col sm:flex-row gap-1 sm:gap-2">
                                <a href="{{ route('admin.doctors.show', $doctor->id) }}" class="text-blue-600 hover:text-blue-900">
                                    <i class="fas fa-eye"></i> <span class="hidden sm:inline">View</span>
                                </a>
                                <a href="{{ route('admin.doctors.edit', $doctor->id) }}" class="text-yellow-600 hover:text-yellow-900">
                                    <i class="fas fa-edit"></i> <span class="hidden sm:inline">Edit</span>
                                </a>
                                @if(!$doctor->is_verified)
                                    <form method="POST" action="{{ route('admin.doctors.verify', $doctor->id) }}" class="inline">
                                        @csrf
                                        <button type="submit" class="text-green-600 hover:text-green-900">
                                            <i class="fas fa-check"></i> <span class="hidden sm:inline">Verify</span>
                                        </button>
                                    </form>
                                @endif
                                <form method="POST" action="{{ route('admin.doctors.toggle-status', $doctor->id) }}" class="inline">
                                    @csrf
                                    <button type="submit" class="text-blue-600 hover:text-blue-900">
                                        <i class="fas fa-toggle-{{ $doctor->is_active ? 'on' : 'off' }}"></i> <span class="hidden sm:inline">{{ $doctor->is_active ? 'Deactivate' : 'Activate' }}</span>
                                    </button>
                                </form>
                                <form method="POST" action="{{ route('admin.doctors.delete', $doctor->id) }}" class="inline" onsubmit="return confirm('Are you sure you want to delete this doctor?')">
                                    @csrf
                                    @method('DELETE')
                                    <button type="submit" class="text-red-600 hover:text-red-900">
                                        <i class="fas fa-trash"></i> <span class="hidden sm:inline">Delete</span>
                                    </button>
                                </form>
                            </div>
                        </td>
                    </tr>
                    @empty
                    <tr>
                        <td colspan="5" class="px-6 py-8 text-center text-gray-500">
                            <i class="fas fa-user-slash text-4xl mb-2"></i>
                            <p>No doctors found matching your criteria.</p>
                            @if(request()->hasAny(['search', 'status']))
                                <a href="{{ route('admin.doctors') }}" class="text-blue-600 hover:text-blue-900 mt-2 inline-block">
                                    Clear filters
                                </a>
                            @endif
                        </td>
                    </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
    </div>
</div>
@endsection

